package connection;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;
import javax.swing.WindowConstants;

/**
 * This utility class gets a username and password. Instantiating the class will
 * cause a dialog box to be displayed prompting the user for the username and
 * password. The getUsername() and getPassword() methods can be called on the
 * resulting object to find out what the user entered.
 * 
 * @author Curt Clifton, January 2006
 */
public class ConnectionInfo implements ActionListener {
	private String username;

	// FIXME: this should really be stored encrypted
	// to avoid having the plaintext in memory for any length of time.  Using a char[]
	// instead of a String provides a (very small) measure of protection.
	private char[] password;

	// records whether initialization is finished (i.e., the dialog is done)
	private transient boolean initialized = false;
	
	private boolean wasCancelled = false;
	
	private Thread callerThread;

	private JTextField userNameTextField;

	private JPasswordField passwordTextField;

	/**
	 * Prompts user for username and password, returning a new instance of this
	 * class after the user is done with the dialog box.
	 */
	public ConnectionInfo() {
		this("Connection Information",
				"Enter your connection information.");
	}

	/**
	 * Prompts user for username and password, returning a new instance of this
	 * class after the user is done with the dialog box.
	 * 
	 * @param title
	 *            for the dialog box
	 * @param prompt
	 *            describing what data is needed
	 */
	public ConnectionInfo(String title, String prompt) {
		promptForInfoWithWait(title, prompt);
	}

	/**
	 * @return the username associated with this connection info
	 */
	public String getUsername() {
		return this.username;
	}

	/**
	 * Returns the password associated with this connection info. Calling code
	 * should avoid storing the result in a field, which would cause the
	 * plaintext password to appear in memory longer than necessary.
	 * 
	 * @return the password associated with this connection info
	 */
	public char[] getPassword() {
		return this.password;
	}

	/**
	 * Displays the dialog box and then hangs until the user is done with the
	 * dialog.
	 * 
	 * @param title
	 *            for the dialog box
	 * @param prompt
	 *            describing what data is needed
	 */
	private void promptForInfoWithWait(String title, String prompt) {
		JDialog diag = new JDialog();
		diag.setTitle(title);
		diag.setResizable(false);
		diag.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		diag.addWindowListener(new WindowAdapter() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see java.awt.event.WindowAdapter#windowClosing(java.awt.event.WindowEvent)
			 */
			@Override
			public void windowClosing(@SuppressWarnings("unused") WindowEvent we) {
				handleCancel();
			}
		});

		GridBagLayout layout = new GridBagLayout();
		GridBagConstraints cons = new GridBagConstraints();
		cons.insets = new Insets(3, 5, 5, 3);
		diag.setLayout(layout);

		JLabel promptLabel = new JLabel(prompt);
		cons.gridx = 0;
		cons.gridy = 0;
		cons.gridwidth = 3;
		addComponent(diag, promptLabel, layout, cons);

		JLabel userNameLabel = new JLabel("Username:");
		cons.gridx = 0;
		cons.gridy = 1;
		cons.gridwidth = 1;
		addComponent(diag, userNameLabel, layout, cons);

		this.userNameTextField = new JTextField();
		cons.gridx = 1;
		cons.gridy = 1;
		cons.gridwidth = 2;
		cons.fill = GridBagConstraints.HORIZONTAL;
		addComponent(diag, this.userNameTextField, layout, cons);

		JLabel passwordLabel = new JLabel("Password:");
		cons.gridx = 0;
		cons.gridy = 2;
		cons.gridwidth = 1;
		cons.fill = GridBagConstraints.NONE;
		addComponent(diag, passwordLabel, layout, cons);

		this.passwordTextField = new JPasswordField();
		cons.gridx = 1;
		cons.gridy = 2;
		cons.gridwidth = 2;
		cons.fill = GridBagConstraints.HORIZONTAL;
		addComponent(diag, this.passwordTextField, layout, cons);

		JButton cancelButton = new JButton("Cancel");
		cancelButton.addActionListener(this);
		cons.gridx = 1;
		cons.gridy = 3;
		cons.gridwidth = 1;
		addComponent(diag, cancelButton, layout, cons);

		JButton okButton = new JButton("OK");
		okButton.addActionListener(this);
		cons.gridx = 2;
		cons.gridy = 3;
		cons.gridwidth = 1;
		addComponent(diag, okButton, layout, cons);
		diag.getRootPane().setDefaultButton(okButton);

		diag.pack();
		diag.setVisible(true);

		// Stashes the current thread so we can wake it up when the user is done
		// with the dialog
		this.callerThread = Thread.currentThread();

		// Waits for the user to complete the dialog
		while (!this.initialized) {
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				// Ignores the exception and lets the while condition check the
				// initialization status
			}
		}

		// Lets Java know we're done with the dialog
		diag.dispose();
	}

	/**
	 * Adds the given component to the given dialog, using the given layout and constraints.
	 * @param diag
	 * @param comp
	 * @param layout
	 * @param cons
	 */
	private void addComponent(JDialog diag, Component comp,
			GridBagLayout layout, GridBagConstraints cons) {
		layout.setConstraints(comp, cons);
		diag.add(comp);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent ev) {
		if (ev.getActionCommand().equals("OK")) {
			this.username = this.userNameTextField.getText();
			this.password = this.passwordTextField.getPassword();
			this.passwordTextField.setText("");
			markAsInitialized();
		} else {
			handleCancel();
		}
	}

	/**
	 * Handles a cancelled dialog, whether by closing or by clicking cancel or
	 * whatever.
	 */
	private void handleCancel() {
		this.username = null;
		this.password = null;
		this.wasCancelled = true;
		markAsInitialized();
	}

	/**
	 * Marks this instance as having been initialized.
	 */
	private synchronized void markAsInitialized() {
		this.initialized = true;
		this.callerThread.interrupt();
	}

	/**
	 * @return true if the user cancelled the request for username and password information
	 */
	public boolean wasCancelled() {
		return this.wasCancelled;
	}
}
