Beyond CPS -- do-it-yourself continuations

One of the things that is unsatisfying about implementing our scheme interpreter in scheme is using scheme procedures to represent continuations.  Here we see how to represent continuations as data structures, then we go a step farther by demonstrating a transformation that could replace recursive calls by mutation of global variables and simple goto statements.

We illustrate the transformations using two simple and familiar procedures, reverse* and append.  While it would be tedious, the same transformations can be applied to your interpreter, as well.

; --file 1-reverse.ss ----------------------------
; --Starting point: original version of reverse* --

(define reverse*
  (lambda (L)
    (if (null? L)
        '()
        (append (reverse* (cdr L))
                (list (if (pair? (car L))
                          (reverse* (car L))
                          (car L)))))))

;(define append     ; this is a comment 
;  (lambda (a b)    ; because append is already 
;    (if (null? a)  ; defined in scheme
;       b
;       (cons (car a)
;             (append (cdr a) b)))))



;------------ file 2-cps.ss ------------------
;------------ a cps version of reverse. -----

(define reverse*-cps
  (lambda (L k)
    (if (null? L)
        (k '())
        (reverse*-cps
         (cdr L)
         (lambda (reversed-cdr)
           (if (pair? (car L))
               (reverse*-cps
                (car L)
                (lambda (reversed-car)
                  (append-cps reversed-cdr
                              (list reversed-car)
                              k)))
               (append-cps reversed-cdr
                           (list (car L))
                           k)))))))

(define append-cps
  (lambda (a b k)
    (if (null? a)
        (k b)
        (append-cps
         (cdr a)
         b
         (lambda (appended-cdr)
           (k (cons (car a) appended-cdr)))))))

(define init-k
  (lambda (v)
    (display "answer: ") (display v) (newline)))



;-----fil2 2-reverse-cps-trace.ss ------
;-----A cps-version with tracing -------

(trace-define reverse*-cps
  (lambda (L k)
    (if (null? L)
        (k '())
        (reverse*-cps
         (cdr L)
         (trace-lambda k-cdr (reversed-cdr)
           (if (pair? (car L))
               (reverse*-cps
                (car L)
                (trace-lambda k-car (reversed-car)
                  (append-cps
                   reversed-cdr
                   (list reversed-car)
                   k)))
               (append-cps reversed-cdr
                           (list (car L))
                           k)))))))

(trace-define append-cps
  (lambda (a b k)
    (if (null? a)
        (k b)
        (append-cps
         (cdr a)
         b
         (trace-lambda k-append (appended-cdr)
            (k (cons (car a) appended-cdr)))))))

(trace-define init-k
  (lambda (v)
    (display "answer: ") (display v) (newline)))


> (load "2-reverse-cps-trace.ss")
> (reverse*-cps '(1 (2 3 (4 ()))) init-k)
|(reverse*-cps (1 (2 3 (4 ()))) #<procedure>)
|(reverse*-cps ((2 3 (4 ()))) #<procedure>)
|(reverse*-cps () #<procedure>)
|(k-cdr ())
|(reverse*-cps (2 3 (4 ())) #<procedure>)
|(reverse*-cps (3 (4 ())) #<procedure>)
|(reverse*-cps ((4 ())) #<procedure>)
|(reverse*-cps () #<procedure>)
|(k-cdr ())
|(reverse*-cps (4 ()) #<procedure>)
|(reverse*-cps (()) #<procedure>)
|(reverse*-cps () #<procedure>)
|(k-cdr ())
|(append-cps () (()) #<procedure>)
|(k-cdr (()))
|(append-cps (()) (4) #<procedure>)
|(append-cps () (4) #<procedure>)
|(k-append (4))
|(k-car (() 4))
|(append-cps () ((() 4)) #<procedure>)
|(k-cdr ((() 4)))
|(append-cps ((() 4)) (3) #<procedure>)
|(append-cps () (3) #<procedure>)
|(k-append (3))
|(k-cdr ((() 4) 3))
|(append-cps ((() 4) 3) (2) #<procedure>)
|(append-cps (3) (2) #<procedure>)
|(append-cps () (2) #<procedure>)
|(k-append (2))
|(k-append (3 2))
|(k-car ((() 4) 3 2))
|(append-cps () (((() 4) 3 2)) #<procedure>)
|(k-cdr (((() 4) 3 2)))
|(append-cps (((() 4) 3 2)) (1) #<procedure>)
|(append-cps () (1) #<procedure>)
|(k-append (1))
|(init-k (((() 4) 3 2) 1))
answer: (((() 4) 3 2) 1)


;----- file 3-reverse-k.ss -------------------
;----- now we treat continuations as an ADT---

; For each kind of continuation, we create a
; "make" procedure.   We also create an
; apply-continuation procedure that knows how to apply
; any continuation from this code.

(define reverse*-k
  (lambda (L k)
    (if (null? L)
        (apply-continuation k '())
        (reverse*-k (cdr L)
                    (make-rev1 k L)))))

(define append-k
  (lambda (a b k)
    (if (null? a)
        (apply-continuation k b)
        (append-k (cdr a) b (make-append-cont k a)))))

; Te above code is independent of the
; representation of continuations.


; Representation #1.
; As we did with finite functions,  we first
; implement continuations as scheme procedures. 

(define make-rev1
  (lambda (k L)
    (lambda (reversed-cdr)
      (if (pair? (car L))
          (reverse*-k (car L)
                      (make-rev2
                       reversed-cdr k))
          (append-k reversed-cdr
                    (list (car L)) k)))))

(define make-rev2
  (lambda  reversed-cdr k)
    (lambda (reversed-car)
      (append-k reversed-cdr
                (list reversed-car) k))))

(define make-init-k
  (lambda ()
    (lambda (v)
      (display "answer: ")
      (display v)
      (newline))))

(define make-append-cont
  (lambda (k a)
    (lambda (appended-cdr)
      (apply-continuation
       k
       (cons (car a) appended-cdr)))))

(define apply-continuation
  (lambda (k v) (k v)))

(define testk
  (lambda () 
    (reverse*-k
     '(1 ((2 3) () (((4)))))
     (make-init-k))))






;---- file 3-reverse-k-trace.ss ----
;---- Same as the previous program, but with tracing

(trace-define reverse*-k
  (lambda (L k)
    (if (null? L)
        (apply-continuation k '())
        (reverse*-k (cdr L)
                    (make-rev1 k L)))))

(trace-define append-k
  (lambda (a b k)
    (if (null? a)
        (apply-continuation k b)
        (append-k (cdr a)
                  b
                  (make-append-cont k a)))))

(trace-define make-rev1
  (lambda (k L)
    (trace-lambda rev1-k (reversed-cdr)
      (if (pair? (car L))
          (reverse*-k
           (car L)
           (make-rev2 reversed-cdr k))
          (append-k reversed-cdr
                    (list (car L))
                    k)))))

(trace-define make-rev2
  (lambda  (reversed-cdr k)
    (trace-lambda rev2-k (reversed-car)
      (append-k reversed-cdr
                (list reversed-car)
                k))))

(trace-define make-init-k
  (lambda ()
    (trace-lambda init-k (v)
      (display "answer: ")
      (display v)
      (newline))))

(trace-define make-append-cont
  (lambda (k a)
    (trace-lambda append-k (appended-cdr)
      (apply-continuation k
                          (cons (car a)
                                appended-cdr)))))

(trace-define apply-continuation
  (lambda (k v) (k v)))

(trace-define testk
  (lambda () 
    (reverse*-k '(1 ((2 3) () 4))
                (make-init-k))))

> (load "3-reverse-k-trace.ss")
> (testk)
|(testk)
| (make-init-k)
| #<procedure>
|(reverse*-k (1 ((2 3) () 4)) #<procedure>)
| (make-rev1 #<procedure> (1 ((2 3) () 4)))
| #<procedure>
|(reverse*-k (((2 3) () 4)) #<procedure>)
| (make-rev1 #<procedure> (((2 3) () 4)))
| #<procedure>
|(reverse*-k () #<procedure>)
|(apply-continuation #<procedure> ())
|(rev1-k ())
| (make-rev2 () #<procedure>)
| #<procedure>
|(reverse*-k ((2 3) () 4) #<procedure>)
| (make-rev1 #<procedure> ((2 3) () 4))
| #<procedure>
|(reverse*-k (() 4) #<procedure>)
| (make-rev1 #<procedure> (() 4))
| #<procedure>
|(reverse*-k (4) #<procedure>)
| (make-rev1 #<procedure> (4))
| #<procedure>
|(reverse*-k () #<procedure>)
|(apply-continuation #<procedure> ())
|(rev1-k ())
|(append-k () (4) #<procedure>)
|(apply-continuation #<procedure> (4))
|(rev1-k (4))
|(append-k (4) (()) #<procedure>)
| (make-append-cont #<procedure> (4))
| #<procedure>
|(append-k () (()) #<procedure>)
|(apply-continuation #<procedure> (()))
|(append-k (()))
|(apply-continuation #<procedure> (4 ()))
|(rev1-k (4 ()))
| (make-rev2 (4 ()) #<procedure>)
| #<procedure>
|(reverse*-k (2 3) #<procedure>)
| (make-rev1 #<procedure> (2 3))
| #<procedure>
|(reverse*-k (3) #<procedure>)
| (make-rev1 #<procedure> (3))
| #<procedure>
|(reverse*-k () #<procedure>)
|(apply-continuation #<procedure> ())
|(rev1-k ())
|(append-k () (3) #<procedure>)
|(apply-continuation #<procedure> (3))
|(rev1-k (3))
|(append-k (3) (2) #<procedure>)
| (make-append-cont #<procedure> (3))
| #<procedure>
|(append-k () (2) #<procedure>)
|(apply-continuation #<procedure> (2))
|(append-k (2))
|(apply-continuation #<procedure> (3 2))
|(rev2-k (3 2))
|(append-k (4 ()) ((3 2)) #<procedure>)
| (make-append-cont #<procedure> (4 ()))
| #<procedure>
|(append-k (()) ((3 2)) #<procedure>)
| (make-append-cont #<procedure> (()))
| #<procedure>
|(append-k () ((3 2)) #<procedure>)
|(apply-continuation #<procedure> ((3 2)))
|(append-k ((3 2)))
|(apply-continuation #<procedure> (() (3 2)))
|(append-k (() (3 2)))
|(apply-continuation #<procedure> (4 () (3 2)))
|(rev2-k (4 () (3 2)))
|(append-k () ((4 () (3 2))) #<procedure>)
|(apply-continuation #<procedure> ((4 () (3 2))))
|(rev1-k ((4 () (3 2))))
|(append-k ((4 () (3 2))) (1) #<procedure>)
| (make-append-cont #<procedure> ((4 () (3 2))))
| #<procedure>
|(append-k () (1) #<procedure>)
|(apply-continuation #<procedure> (1))
|(append-k (1))
|(apply-continuation #<procedure> ((4 () (3 2)) 1))
|(init-k ((4 () (3 2)) 1))
answer: ((4 () (3 2)) 1)
|#<void>



;--- file 4-reverse-k-data-structure.ss -------------------
;--- implement the continuations ADT as data-structures ---
;--- use the same definitions of reverse-k and append-k ---
;--- as in 3-reverse-k.ss ---------------------------------

(load "record.ss")
(define-record append-k (k a))
(define-record init-k ())
(define-record rev1-k (k L))
(define-record rev2-k (reversed-cdr k))

(define apply-continuation
  (lambda (k v)
    (variant-case k
      [init-k ()
           (display "answer: ")
           (display v)
           (newline)]
      [append-k (k a)
        (apply-continuation k
                            (cons (car a)
                                  v))]
      [rev1-k (k L)
        (if (pair? (car L))
            (reverse*-k (car L)
                        (make-rev2-k v k))
            (append-k v
                      (list (car L))
                      k))]
      [rev2-k (reversed-cdr k)
              (append-k
               reversed-cdr
               (list v)
               k)])))


(define reverse*-k
  (lambda (L k)
    (if (null? L)
        (apply-continuation k '())
        (reverse*-k (cdr L)
                    (make-rev1-k k L)))))

(define append-k
  (lambda (a b k)
    (if (null? a)
        (apply-continuation k b)
        (append-k (cdr a)
                  b
                  (make-append-k k a)))))

(define testk
  (lambda () 
    (reverse*-k '(1 ((2 3) () (((4)))))
                (make-init-k))))



;--- file 4-reverse-k-data-structure.ss -------------------
;--- implement the continuations ADT as data-structures ---
;--- use the same definitions of reverse-k and append-k ---
;--- as in 3-reverse-k.ss ---------------------------------

(load "record.ss")
(define-record append-k (k a))
(define-record init-k ())
(define-record rev1-k (k L))
(define-record rev2-k (reversed-cdr k))

(trace-define apply-continuation
  (lambda (k v)
    (variant-case k
      [init-k ()
           (display "answer: ")
           (display v)
           (newline)]
      [append-k (k a)
        (apply-continuation k
                            (cons (car a)
                                  v))]
      [rev1-k (k L)
        (if (pair? (car L))
            (reverse*-k (car L)
                        (make-rev2-k v k))
            (append-k v
                      (list (car L))
                      k))]
      [rev2-k (reversed-cdr k)
              (append-k
               reversed-cdr
               (list v)
               k)])))

(trace-define reverse*-k
  (lambda (L k)
    (if (null? L)
        (apply-continuation k '())
        (reverse*-k (cdr L)
                    (make-rev1-k k L)))))

(trace-define append-k
  (lambda (a b k)
    (if (null? a)
        (apply-continuation k b)
        (append-k (cdr a)
                  b
                  (make-append-k k a)))))

(define testk
  (lambda () 
    (reverse*-k '(1 ((2 3) () (((4)))))
                (make-init-k))))



> (load "4-reverse-k-data-structure-trace.ss")
> (testk)
|(reverse*-k (1 ((2 3) () (((4))))) #1(init-k))
|(reverse*-k (((2 3) () (((4))))) #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
|(reverse*-k
   ()
   #3(rev1-k #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))) (((2 3) () (((4)))))))
|(apply-continuation
   #3(rev1-k #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))) (((2 3) () (((4))))))
   ())
|(reverse*-k
   ((2 3) () (((4))))
   #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
|(reverse*-k
   (() (((4))))
   #3(rev1-k
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
      ((2 3) () (((4))))))
|(reverse*-k
   ((((4))))
   #3(rev1-k
      #3(rev1-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((2 3) () (((4)))))
      (() (((4))))))
|(reverse*-k
   ()
   #3(rev1-k
      #3(rev1-k
         #3(rev1-k
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
            ((2 3) () (((4)))))
         (() (((4)))))
      ((((4))))))
|(apply-continuation
   #3(rev1-k
      #3(rev1-k
         #3(rev1-k
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
            ((2 3) () (((4)))))
         (() (((4)))))
      ((((4)))))
   ())
|(reverse*-k
   (((4)))
   #3(rev2-k
      ()
      #3(rev1-k
         #3(rev1-k
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
            ((2 3) () (((4)))))
         (() (((4)))))))
|(reverse*-k
   ()
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev1-k
            #3(rev1-k
               #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
               ((2 3) () (((4)))))
            (() (((4))))))
      (((4)))))
|(apply-continuation
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev1-k
            #3(rev1-k
               #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
               ((2 3) () (((4)))))
            (() (((4))))))
      (((4))))
   ())
|(reverse*-k
   ((4))
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev1-k
            #3(rev1-k
               #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
               ((2 3) () (((4)))))
            (() (((4))))))))
|(reverse*-k
   ()
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev1-k
               #3(rev1-k
                  #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                  ((2 3) () (((4)))))
               (() (((4)))))))
      ((4))))
|(apply-continuation
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev1-k
               #3(rev1-k
                  #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                  ((2 3) () (((4)))))
               (() (((4)))))))
      ((4)))
   ())
|(reverse*-k
   (4)
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev1-k
               #3(rev1-k
                  #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                  ((2 3) () (((4)))))
               (() (((4)))))))))
|(reverse*-k
   ()
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev2-k
               ()
               #3(rev1-k
                  #3(rev1-k
                     #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                     ((2 3) () (((4)))))
                  (() (((4))))))))
      (4)))
|(apply-continuation
   #3(rev1-k
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev2-k
               ()
               #3(rev1-k
                  #3(rev1-k
                     #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                     ((2 3) () (((4)))))
                  (() (((4))))))))
      (4))
   ())
|(append-k
   ()
   (4)
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev1-k
               #3(rev1-k
                  #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                  ((2 3) () (((4)))))
               (() (((4)))))))))
|(apply-continuation
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev2-k
            ()
            #3(rev1-k
               #3(rev1-k
                  #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
                  ((2 3) () (((4)))))
               (() (((4))))))))
   (4))
|(append-k
   ()
   ((4))
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev1-k
            #3(rev1-k
               #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
               ((2 3) () (((4)))))
            (() (((4))))))))
|(apply-continuation
   #3(rev2-k
      ()
      #3(rev2-k
         ()
         #3(rev1-k
            #3(rev1-k
               #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
               ((2 3) () (((4)))))
            (() (((4)))))))
   ((4)))
|(append-k
   ()
   (((4)))
   #3(rev2-k
      ()
      #3(rev1-k
         #3(rev1-k
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
            ((2 3) () (((4)))))
         (() (((4)))))))
|(apply-continuation
   #3(rev2-k
      ()
      #3(rev1-k
         #3(rev1-k
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
            ((2 3) () (((4)))))
         (() (((4))))))
   (((4))))
|(append-k
   ()
   ((((4))))
   #3(rev1-k
      #3(rev1-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((2 3) () (((4)))))
      (() (((4))))))
|(apply-continuation
   #3(rev1-k
      #3(rev1-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((2 3) () (((4)))))
      (() (((4)))))
   ((((4)))))
|(append-k
   ((((4))))
   (())
   #3(rev1-k
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
      ((2 3) () (((4))))))
|(append-k
   ()
   (())
   #3(append-k
      #3(rev1-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((2 3) () (((4)))))
      ((((4))))))
|(apply-continuation
   #3(append-k
      #3(rev1-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((2 3) () (((4)))))
      ((((4)))))
   (()))
|(apply-continuation
   #3(rev1-k
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
      ((2 3) () (((4)))))
   ((((4))) ()))
|(reverse*-k
   (2 3)
   #3(rev2-k
      ((((4))) ())
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))))
|(reverse*-k
   (3)
   #3(rev1-k
      #3(rev2-k
         ((((4))) ())
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
      (2 3)))
|(reverse*-k
   ()
   #3(rev1-k
      #3(rev1-k
         #3(rev2-k
            ((((4))) ())
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
         (2 3))
      (3)))
|(apply-continuation
   #3(rev1-k
      #3(rev1-k
         #3(rev2-k
            ((((4))) ())
            #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
         (2 3))
      (3))
   ())
|(append-k
   ()
   (3)
   #3(rev1-k
      #3(rev2-k
         ((((4))) ())
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
      (2 3)))
|(apply-continuation
   #3(rev1-k
      #3(rev2-k
         ((((4))) ())
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
      (2 3))
   (3))
|(append-k
   (3)
   (2)
   #3(rev2-k
      ((((4))) ())
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))))
|(append-k
   ()
   (2)
   #3(append-k
      #3(rev2-k
         ((((4))) ())
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
      (3)))
|(apply-continuation
   #3(append-k
      #3(rev2-k
         ((((4))) ())
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
      (3))
   (2))
|(apply-continuation
   #3(rev2-k
      ((((4))) ())
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
   (3 2))
|(append-k
   ((((4))) ())
   ((3 2))
   #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))))
|(append-k
   (())
   ((3 2))
   #3(append-k
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
      ((((4))) ())))
|(append-k
   ()
   ((3 2))
   #3(append-k
      #3(append-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((((4))) ()))
      (())))
|(apply-continuation
   #3(append-k
      #3(append-k
         #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
         ((((4))) ()))
      (()))
   ((3 2)))
|(apply-continuation
   #3(append-k
      #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
      ((((4))) ()))
   (() (3 2)))
|(apply-continuation
   #3(rev2-k () #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
   ((((4))) () (3 2)))
|(append-k () (((((4))) () (3 2))) #3(rev1-k #1(init-k) (1 ((2 3) () (((4)))))))
|(apply-continuation
   #3(rev1-k #1(init-k) (1 ((2 3) () (((4))))))
   (((((4))) () (3 2))))
|(append-k (((((4))) () (3 2))) (1) #1(init-k))
|(append-k () (1) #3(append-k #1(init-k) (((((4))) () (3 2)))))
|(apply-continuation #3(append-k #1(init-k) (((((4))) () (3 2)))) (1))
|(apply-continuation #1(init-k) (((((4))) () (3 2)) 1))
answer: (((((4))) () (3 2)) 1)
|#<void>




; -- file 6-reverse-imperative.ss ----------------------
; -- imperative version of  reverse* --------------------
; -- This puts it into a form where all of the ---------
; -- recursion can be replaced by assignments and "goto"s

(load "record.ss")  
(define-record append-cont (a k1))
(define-record init-k ())
(define-record rev1 (k1 car-L)) 
(define-record rev2 (reversed-cdr k1))

(define reverse*
  (lambda (L)
    (let ([L L]
          [k (make-init-k)]
          [a '*unbound]
          [b '*unbound]
          [v '*unbound*])
      (letrec
          ([reverse*
              (lambda ()
                (if (null? L)
                    (begin
                      (set! v '())
                      (apply-continuation))
                    (begin 
                      (set! k
                            (make-rev1 k
                                       (car L)))
                    (set! L (cdr L))
                    (reverse*))))]
           [append
               (lambda ()
                 (if (null? a)
                     (begin
                       (set! v b)
                       (apply-continuation))
                     (begin
                       (set! k
                             (make-append-cont a
                                               k))
                       (set! a (cdr a))
                       (append))))]
           [apply-continuation
                (lambda ()  
                  (variant-case k
                    [init-k ()
                      (display "answer: ")
                      (display v)
                      (newline)]
                    [append-cont (a k1)
                       (set! v (cons (car a) v))
                       (set! k k1)
                       (apply-continuation)]
                    [rev1 (k1 car-L)
                       (if (pair? car-L)
                           (begin
                             (set! L car-L)
                             (set! k
                               (make-rev2 v k1))
                             (reverse*))
                           (begin
                             (set! a v)
                             (set! b (list car-L))
                             (set! k k1)
                             (append)))]
                    [rev2 (reversed-cdr k1)
                       (set! a reversed-cdr)
                       (set! b (list v))
                       (set! k k1)
                       (append)]))])
        (reverse*)))))


(define testk
    (lambda ()
          (reverse*
           '(1 ((2 3) () (((4))))))))




;-- file 6-reverse-imperative-trace.ss ---
;-- add tracing to previous code.

(load "record.ss")  
(define-record append-cont (a k1)) 
(define-record init-k ())
(define-record rev1 (k1 car-L))  
(define-record rev2 (reversed-cdr k1))

(define *tracing*)

(define reverse*
  (lambda (L)
    (let ([L L]
          [k (make-init-k)]
          [a '*unbound]
          [b '*unbound]
          [v '*unbound])
      (letrec
          ([traceit
              (lambda (sym)
                (when (top-level-bound? '*tracing*)
                  (printf "~a " sym)
                  (printf "L=~s" L)
                  (printf "  a=~s" a)
                  (printf "  b=~s" b)
                  (printf "  v=~s~%" v)
                  (printf "           k=~s~%" k)))]
           [reverse*
              (lambda ()
                (traceit "reverse*  ")
                (if (null? L)
                    (begin
                      (set! v '())
                      (apply-continuation))
                    (begin 
                      (set! k (make-rev1 k (car L)))
                      (set! L (cdr L))
                      (reverse*))))]
           [append
              (lambda ()
                (traceit "append    ")
                (if (null? a)
                    (begin
                      (set! v b)
                      (apply-continuation))
                    (begin
                      (set! k (make-append-cont a k))
                      (set! a (cdr a))
                      (append))))]
           [apply-continuation
              (lambda ()
                (traceit "apply-cont")
                (variant-case k
                  [init-k ()
                     (display "answer: ")
                     (display v)
                     (newline)]
                  [append-cont (a k1)
                     (set! v (cons (car a) v))
                     (set! k k1)
                     (apply-continuation)]
                  [rev1 (k1 car-L)
                     (if (pair? car-L)
                         (begin
                           (set! L car-L)
                           (set! k (make-rev2  v k1))
                           (reverse*))
                         (begin
                           (set! a v)
                           (set! b (list car-L))
                           (set! k k1)
                           (append)))]
                  [rev2 (reversed-cdr k1)
                     (set! a reversed-cdr)
                     (set! b (list v))
                     (set! k k1)
                     (append)]))])
        (reverse*)))))


> (reverse* '((1 2) 4))
reverse*   L=((1 2) 4)  a=*unbound  b=*unbound  v=*unbound
           k=#1(init-k)
reverse*   L=(4)  a=*unbound  b=*unbound  v=*unbound
           k=#3(rev1 #1(init-k) (1 2))
reverse*   L=()  a=*unbound  b=*unbound  v=*unbound
           k=#3(rev1 #3(rev1 #1(init-k) (1 2)) 4)
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev1 #1(init-k) (1 2)) 4)
append     L=()  a=()  b=(4)  v=()
           k=#3(rev1 #1(init-k) (1 2))
apply-cont L=()  a=()  b=(4)  v=(4)
           k=#3(rev1 #1(init-k) (1 2))
reverse*   L=(1 2)  a=()  b=(4)  v=(4)
           k=#3(rev2 (4) #1(init-k))
reverse*   L=(2)  a=()  b=(4)  v=(4)
           k=#3(rev1 #3(rev2 (4) #1(init-k)) 1)
reverse*   L=()  a=()  b=(4)  v=(4)
           k=#3(rev1 #3(rev1 #3(rev2 (4) #1(init-k)) 1) 2)
apply-cont L=()  a=()  b=(4)  v=()
           k=#3(rev1 #3(rev1 #3(rev2 (4) #1(init-k)) 1) 2)
append     L=()  a=()  b=(2)  v=()
           k=#3(rev1 #3(rev2 (4) #1(init-k)) 1)
apply-cont L=()  a=()  b=(2)  v=(2)
           k=#3(rev1 #3(rev2 (4) #1(init-k)) 1)
append     L=()  a=(2)  b=(1)  v=(2)
           k=#3(rev2 (4) #1(init-k))
append     L=()  a=()  b=(1)  v=(2)
           k=#3(append-cont (2) #3(rev2 (4) #1(init-k)))
apply-cont L=()  a=()  b=(1)  v=(1)
           k=#3(append-cont (2) #3(rev2 (4) #1(init-k)))
apply-cont L=()  a=()  b=(1)  v=(2 1)
           k=#3(rev2 (4) #1(init-k))
append     L=()  a=(4)  b=((2 1))  v=(2 1)
           k=#1(init-k)
append     L=()  a=()  b=((2 1))  v=(2 1)
           k=#3(append-cont (4) #1(init-k))
apply-cont L=()  a=()  b=((2 1))  v=((2 1))
           k=#3(append-cont (4) #1(init-k))
apply-cont L=()  a=()  b=((2 1))  v=(4 (2 1))
           k=#1(init-k)
answer: (4 (2 1))
()
> 

Another more complex example:

> (load "6-reverse-imperative-trace.ss")
> (testk)
reverse*   L=(1 ((2 3) () (((4)))))  a=*unbound  b=*unbound  v=*unbound
           k=#1(init-k)
reverse*   L=(((2 3) () (((4)))))  a=*unbound  b=*unbound  v=*unbound
           k=#3(rev1 #1(init-k) 1)
reverse*   L=()  a=*unbound  b=*unbound  v=*unbound
           k=#3(rev1 #3(rev1 #1(init-k) 1) ((2 3) () (((4)))))
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev1 #1(init-k) 1) ((2 3) () (((4)))))
reverse*   L=((2 3) () (((4))))  a=*unbound  b=*unbound  v=()
           k=#3(rev2 () #3(rev1 #1(init-k) 1))
reverse*   L=(() (((4))))  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3))
reverse*   L=((((4))))  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())
reverse*   L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()) (((4))))
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()) (((4))))
reverse*   L=(((4)))  a=*unbound  b=*unbound  v=()
           k=#3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))
reverse*   L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())) ((4)))
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())) ((4)))
reverse*   L=((4))  a=*unbound  b=*unbound  v=()
           k=#3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())))
reverse*   L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))) (4))
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))) (4))
reverse*   L=(4)  a=*unbound  b=*unbound  v=()
           k=#3(rev2 () #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))))
reverse*   L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())))) 4)
apply-cont L=()  a=*unbound  b=*unbound  v=()
           k=#3(rev1 #3(rev2 () #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())))) 4)
append     L=()  a=()  b=(4)  v=()
           k=#3(rev2 () #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))))
apply-cont L=()  a=()  b=(4)  v=(4)
           k=#3(rev2 () #3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))))
append     L=()  a=()  b=((4))  v=(4)
           k=#3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())))
apply-cont L=()  a=()  b=((4))  v=((4))
           k=#3(rev2 () #3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())))
append     L=()  a=()  b=(((4)))  v=((4))
           k=#3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))
apply-cont L=()  a=()  b=(((4)))  v=(((4)))
           k=#3(rev2 () #3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ()))
append     L=()  a=()  b=((((4))))  v=(((4)))
           k=#3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())
apply-cont L=()  a=()  b=((((4))))  v=((((4))))
           k=#3(rev1 #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)) ())
append     L=()  a=((((4))))  b=(())  v=((((4))))
           k=#3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3))
append     L=()  a=()  b=(())  v=((((4))))
           k=#3(append-cont ((((4)))) #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)))
apply-cont L=()  a=()  b=(())  v=(())
           k=#3(append-cont ((((4)))) #3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3)))
apply-cont L=()  a=()  b=(())  v=((((4))) ())
           k=#3(rev1 #3(rev2 () #3(rev1 #1(init-k) 1)) (2 3))
reverse*   L=(2 3)  a=()  b=(())  v=((((4))) ())
           k=#3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1)))
reverse*   L=(3)  a=()  b=(())  v=((((4))) ())
           k=#3(rev1 #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))) 2)
reverse*   L=()  a=()  b=(())  v=((((4))) ())
           k=#3(rev1 #3(rev1 #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))) 2) 3)
apply-cont L=()  a=()  b=(())  v=()
           k=#3(rev1 #3(rev1 #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))) 2) 3)
append     L=()  a=()  b=(3)  v=()
           k=#3(rev1 #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))) 2)
apply-cont L=()  a=()  b=(3)  v=(3)
           k=#3(rev1 #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))) 2)
append     L=()  a=(3)  b=(2)  v=(3)
           k=#3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1)))
append     L=()  a=()  b=(2)  v=(3)
           k=#3(append-cont (3) #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))))
apply-cont L=()  a=()  b=(2)  v=(2)
           k=#3(append-cont (3) #3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))))
apply-cont L=()  a=()  b=(2)  v=(3 2)
           k=#3(rev2 ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1)))
append     L=()  a=((((4))) ())  b=((3 2))  v=(3 2)
           k=#3(rev2 () #3(rev1 #1(init-k) 1))
append     L=()  a=(())  b=((3 2))  v=(3 2)
           k=#3(append-cont ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1)))
append     L=()  a=()  b=((3 2))  v=(3 2)
           k=#3(append-cont (()) #3(append-cont ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))))
apply-cont L=()  a=()  b=((3 2))  v=((3 2))
           k=#3(append-cont (()) #3(append-cont ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1))))
apply-cont L=()  a=()  b=((3 2))  v=(() (3 2))
           k=#3(append-cont ((((4))) ()) #3(rev2 () #3(rev1 #1(init-k) 1)))
apply-cont L=()  a=()  b=((3 2))  v=((((4))) () (3 2))
           k=#3(rev2 () #3(rev1 #1(init-k) 1))
append     L=()  a=()  b=(((((4))) () (3 2)))  v=((((4))) () (3 2))
           k=#3(rev1 #1(init-k) 1)
apply-cont L=()  a=()  b=(((((4))) () (3 2)))  v=(((((4))) () (3 2)))
           k=#3(rev1 #1(init-k) 1)
append     L=()  a=(((((4))) () (3 2)))  b=(1)  v=(((((4))) () (3 2)))
           k=#1(init-k)
append     L=()  a=()  b=(1)  v=(((((4))) () (3 2)))
           k=#3(append-cont (((((4))) () (3 2))) #1(init-k))
apply-cont L=()  a=()  b=(1)  v=(1)
           k=#3(append-cont (((((4))) () (3 2))) #1(init-k))
apply-cont L=()  a=()  b=(1)  v=(((((4))) () (3 2)) 1)
           k=#1(init-k)
answer: (((((4))) () (3 2)) 1)
>