import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.BoxLayout;
import javax.swing.JPanel;


/**
 * This creates the panel which holds the tools and the CurrentColor.
 *
 * @author Jared Mathis.
 *         Created Apr 13, 2008.
 */
public class ToolPanel extends JPanel implements MouseListener {
	
	/**
	 * This is a string list of the file name for the graphics of the tools.  This is protected so that ToolButton can view them.
	 */
	protected static final String[] toolIcons = {
		"pen",
		"line",
		"freeformBox",
		"rectangle",
		"fill",
		"eyedrop",
		"oval",
		"roundbox",
		"paintbrush",
		
// To be implemented?
//		"freeform",
//		"rectangleGrab",
//		"eraser",
//		"fill",
//		"zoom",
//		"spraypaint",
//		"letterbox",
//		"squigglyLine",
		};

	/**
	 * Number of rows of tools.  2 works well.  Other ones work sometimes; depends on how many tools there are.  
	 * These next three fields are protected so that other classes can adjust to the sizes and amounts of this class
	 * and resize accordingly.
	 */
	protected static final int COLS = 2;
	/**
	 * Calculates number of rows based on columns of tools.
	 */
	protected static final int ROWS = toolIcons.length/COLS + (toolIcons.length%COLS > 0 ? 1 : 0);
	
	/**
	 * Spacing between tools.
	 */
	protected static final int SPACING = 5;
	
	/**
	 * Draw Tool.  These fields are visible so that other classes can refer to constants
	 * i.e. if this.tool = ToolPanel.DRAW_TOOL do { this } instead of trying to actually
	 * remember the numbers.  Theie numbers are the order they were implemented.
	 */
	protected static final int DRAW_TOOL = 0;
	
	/**
	 * Line tool
	 */
	protected static final int LINE_TOOL = 1;
	
	/**
	 * Polygonal tool.
	 */
	protected static final int POLY_TOOL = 2;
	
	/**
	 * Rectangle tool.
	 */
	protected static final int RECT_TOOL = 3;
	
	/**
	 * Fill tool.  Does not work well unless you increase your JVM stack.
	 */
	protected static final int FILL_TOOL = 4;
	
	/**
	 * Eyedrop tool.
	 */
	protected static final int EYEDROP_TOOL = 5;
	
	/**
	 * Oval tool.
	 */
	protected static final int OVAL_TOOL = 6;
	
	/**
	 * Round Rectangle tool
	 */
	protected static final int ROUNDBOX_TOOL = 7;
	
	/**
	 * Brush tool
	 */
	protected static final int BRUSH_TOOL = 8;

	private Paint paint;
	/**
	 * The CurrentColor viewer of this ToolPanel.  This is protected so that ColorChooser has see it.
	 */
	protected CurrentColor viewer;
	/**
	 * The panel which holds the tools.
	 */
	protected JPanel toolPanel;

	/**
	 * This is protected so that ToolButton can modify it.
	 */
	protected int counter = 0;

	/**
	 * The ToolOptions that sometimes appears under the color viewer.  This is protected so that the toolButtons can access it.
	 */
	protected ToolOptions options;
	
	/**
	 * Constructs a ToolPanel, not to be confused with the toolPanel above (The reason the names are the same is because originally there was not
	 * CurrentColor viewer and so a "real" tool panel had to be created. In retrospect, this class should have been named "SidePanel" or something.
	 * 
	 * @param paint The Paint object that mutually has this ToolPanel.
	 */
	public ToolPanel(Paint paint){
		super();		
		BoxLayout layout = new BoxLayout(this, BoxLayout.Y_AXIS);
		this.setLayout(layout);
		
		this.paint = paint;
		
		GridLayout toolsLayout = new GridLayout(ROWS, COLS);
		toolsLayout.setHgap(SPACING);
		toolsLayout.setVgap(SPACING);
		
		// This is the thing that holds the tools
		this.toolPanel = new JPanel();
		this.toolPanel.setLayout(toolsLayout);
		
		// This is the little two-color thing that tells you what color you have selected
		this.viewer = new CurrentColor(this.paint);
		this.viewer.setAlignmentX(Component.CENTER_ALIGNMENT);
		this.viewer.setAlignmentY(Component.TOP_ALIGNMENT);	
		this.viewer.addMouseListener(this.paint.colorChooser);
		
		this.options = new ToolOptions(this.paint);
		
		this.add(this.toolPanel);
		this.add(this.viewer);
		this.add(this.options);
		
		ToolButton tool = new ToolButton(this);
		tool.select();
		
		for (int i = 0; i < toolIcons.length - 1; i++)
			new ToolButton(this);
		}


	public void mouseClicked(MouseEvent e) { /* empty */ }
	public void mouseEntered(MouseEvent e) { /* empty */ }
	public void mouseReleased(MouseEvent e) { /* empty */ }
	public void mouseExited(MouseEvent e) { /* empty */ }

	public void mousePressed(MouseEvent e) {
		Object object = e.getSource();
		if (!(object instanceof ToolButton)) return;
		ToolButton tool = (ToolButton)object;
		
		this.paint.previousTool = this.paint.currentTool;
		this.paint.currentTool = tool.tool;
		tool.select();
		
		this.endCurrentTool();
	}
	
	/**
	 * This ends the current tool which basically clears any overlay the user has drawn (the
	 * temporary image), and makes it so tha paint no longer things a tool is being used.  It
	 * is made protected so that an OptionButton can see it.  Theoretically, a user could 
	 * click once to initiate a tool, move the mouse around and before clicking a second time
	 * to end the tool, they could change the Tool Style or Size in the middle of using a tool,
	 * so ToolButton calls this method when it's clicked so thtat weird things don't happen.
	 * 
	 * It's also used by this class when someone switches tools.
	 * 
	 * It's also used by PaintMenu when someone clicks on the menu bar.
	 * 
	 * It's also used by ColorPanel when someone changes color in the middle of using a tool.
	 * 
	 * It's also used by ColorChooser if someone clicks one of the two colors under the tools.
	 */
	protected void endCurrentTool() {
		this.paint.drawSpace.clearDrawOverlay();
		this.paint.drawSpace.clearDrawOverlay2();
		this.paint.drawSpace.clearFillOverlay();
		this.paint.drawSpace.repaint();
		this.paint.drawSpace.isUsingATool = false;
	}

	
}
