import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

/**
 * This class implements a basic "Paint" program.  Essentially the way the program is structured is 
 * everything "HAS-A" a Paint and to access other parts of paint (i.e. the DrawSpace knowing what tool 
 * is selected or colors are selected) is that class goes through its "this.paint." 
 * 
 * NOTE: The fill tool uses a recursive alogrithm that is prone to make one's JVM run out of stack
 * space.  It is reccomended that you use a VM argument like "-Xss60m" to increase the memory allocated
 * to stack, or only fill tiny areas.
 * 
 * Also, NOTE that the tools are generally initiated with a mouse CLICK (not press) and end with a second 
 * mouse click.  The first mouse click will determine the color (i.e. LMB => primary color, RMB => second-
 * ary color).
 * 
 * Not readily obvious is the JColorChooser activated by clicking on either of the two colors (primary
 * and secondary) on the left underneath all the tool icons.
 * 
 *
 * @author Benjamin Waters, Jared Mathis.
 *         Started work 27 March 2008
 *         Completed 15 April 2008
 */
public class Paint {

	private static final int 
	DEFAULT_HEIGHT = 600, 
	DEFAULT_WIDTH = 600,
	DEFAULT_IMAGE_HEIGHT = 400, 
	DEFAULT_IMAGE_WIDTH = 400;
	

	private static final Color DEFAULT_BACKGROUND_COLOR = Color.WHITE;
	
	
	/**
	 * This will how the current tool will be determined.  Refer to ToolPanel to see what tool each number refers to (i.e. 0 = DrawTool).
	 */
	protected int currentTool; 
	/**
	 * This is the previous tool used so that eye dropper knows what tool to go back to when it's done.
	 */
	protected int previousTool;
	
	/**
	 * The primary color; often used with LMB.
	 */
	protected Color primaryColor = Color.black;
	
	/**
	 * The secondar color; often used with RMB
	 */
	protected Color secondaryColor = Color.white;

	/**
	 * The current tool size.  Only some tools use this.  To view all size possibilities, see the tool sizes and styles declarations in
	 * ToolOptions.
	 */
	protected int currentToolSize;
	
	/**
	 * The current tool style (fill or edge).  Only some tools use this.
	 */
	protected int currentToolStyle;
	
	/**
	 * The window/frame that this Paint creates.
	 */
	protected JFrame window;
	
	/**
	 * The panel of this paint, held by the window.  It holds the DrawSpace, the ToolPanel and the ColorPanel.
	 */
	protected JPanel panel;
	
	/**
	 * The drawable area; this is where all of the magic happens
	 */
	protected DrawSpace drawSpace;
	
	/**
	 * This panel holds the tools, the 
	 */
	protected ToolPanel toolPanel;
	
	/**
	 * The ImageResizer Dialogue box of this paint.  Resizes drawSpace
	 */
	protected ImageResizer resizer;
	
	/**
	 * The FileChooser Frame for this paint.  Let's one open and save files.
	 */
	protected JFileChooser fileChooser;
	
	/**
	 * This is technically what's in Paint's panel field, not drawSpace; this holds a DrawSpace.
	 */
	protected JScrollPane scroller;
	
	/**
	 * This is the ColorChooser Frame that pops up when you click one of the two selected colors below the toolbar.
	 */
	protected ColorChooser colorChooser;

	/**
	 * This is the file menu at the top.  It's protected so that drawSpace can access the enable/disable of the undo buttons.
	 */
	protected PaintMenu menu;

	
	/**
	 * Constructor of a "Paint" object.  Takes no parameters.
	 */
	public Paint() {
		this(DEFAULT_WIDTH, DEFAULT_HEIGHT);
	}
	

	/**
	 * Constructs a "Paint" object.
	 *
	 * @param width Window Width
	 * @param height Window Height
	 */
	public Paint(int width, int height) {
		this.window = new JFrame();
		this.window.setSize(width, height);
		
		this.scroller = new JScrollPane();
		
		this.drawSpace = new DrawSpace(DEFAULT_BACKGROUND_COLOR, this); 
		this.drawSpace.addMouseListener(this.drawSpace);
		this.drawSpace.addMouseMotionListener(this.drawSpace);
		this.drawSpace.setAlignmentY(Component.TOP_ALIGNMENT);
		this.drawSpace.setImageSize(DEFAULT_IMAGE_WIDTH, DEFAULT_IMAGE_HEIGHT);
		this.scroller.setViewportView(this.drawSpace);
		this.drawSpace.setInitiated(true);
		
		this.colorChooser = new ColorChooser(this);
		
		this.toolPanel = new ToolPanel(this);
		
		ColorPanel colorPanel = new ColorPanel(this);
		
		this.resizer = new ImageResizer(this);
		
		this.fileChooser = new JFileChooser();
		this.fileChooser.setFileHidingEnabled(true);
		
		FlowLayout layout = new FlowLayout();
		
		this.panel = new JPanel(layout);
		this.panel.add(this.toolPanel);
		this.panel.add(this.scroller);
		this.panel.add(colorPanel);
		
		this.window.add(this.panel);		
		this.window.setJMenuBar(this.menu = new PaintMenu(this));
		this.window.addComponentListener(this.drawSpace);
		this.window.setTitle("Painter!!");
		this.window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		this.window.setVisible(true);
		
	}	
	
	/**
	 * This method opens up paint.  There will not be much code here so that paint could be opened from another package.
	 *
	 * @param args  Command-line arguments
	 */
	public static void main(String[] args) {
		new Paint();

	}

}
