package ballworlds.ball;

import java.awt.Color;
import java.awt.geom.Point2D;

import ballworlds.framework.BallEnvironment;

/**
 * A Mover starts in the exact middle of its world (even if its world is resized).
 * Each Mover has its own fixed velocity that is set at random
 * when the Mover is constructed. Additionally, the Mover is "selectable",
 * "draggable", and "killable" by the mouse.
 *
 * @author David Mutchler. Created March 20, 2009.
 */
public class Mover extends DudThatMoves {
	private static Point2D MOVER_MAX_SPEED = new Point2D.Double(1.5, 1.0);
	private static Color MOVER_COLOR = Color.blue;
	
	private boolean isPaused;
	
	/**
	 * Adds the Mover to its World at the middle of the World
	 * with the default color and diameter for a Mover,
	 * and with a constant but randomly chosen velocity.
	 *
	 * @param ballEnvironment the object that manages Balls in this World
	 */
	public Mover(BallEnvironment ballEnvironment) {
		super(ballEnvironment);

		this.setLocation(new Point2D.Double(
				ballEnvironment.middleOfWorld().getX(),
				ballEnvironment.middleOfWorld().getY()));
		
		this.setVelocity(new Point2D.Double(
				Mover.MOVER_MAX_SPEED.getX() * (Math.random() * 2 - 1),
				Mover.MOVER_MAX_SPEED.getY() * (Math.random() * 2 - 1)));
				
		this.setColor(Mover.MOVER_COLOR);
		
		this.isPaused = false;
	}
	
	@Override
	public double distanceFrom(Point2D point) {
		return point.distance(this.getLocation());
	}

	@Override
	public void moveTo(Point2D point) {
		this.setLocation(point);
	}

	/**
	 * Act unless the Mover is paused.
	 */
	@Override
	public void act() {
		if (!this.isPaused) {
			super.act();
		}
	}
	
	@Override
	public void die() {
		this.getBallEnvironment().removeBall(this);
	}

	@Override
	public void pauseOrResume() {
		this.isPaused = !this.isPaused;
	}
	
	/**
	 * Returns the value of the field called 'isPaused'.
	 * @return Returns the isPaused.
	 */
	protected final boolean isPaused() {
		return this.isPaused;
	}
}
