function Q = solve_with_diri(mat,rhs,diri_list,ndof)
% usage Q = solve_with_diri(mat,rhs,diri_list,ndof)
% 
% Solves the matrix mat with right hand side rhs constrained by Dirichlet 
% conditions described in diri_list, using Matlab's standard direct sparse
% solver.  For nodal degrees of freedom (ndof) greater than 1, the
% routine reorders to reduce the matrix bandwidth.  The routine enforces 
% Dirichlet conditions by reordering to partition out Dirichlet degrees
% of freedom.  
%
% The solution Q is reported with the original ordering restored.  Q will 
% contain ndof columns, one for each nodal variable.
%
% diri_list should have two columns total and one row for each diri degree
% of freedom.  The first column of each row must contain 
% (diri_node# + (var#-1)*nnodes).  
% The second column contains the actual diri value.
% Example: Suppose that the state varibles alpha and beta are being solved
% for at each node, and there are 100 nodes total.  Let alpha be nodal
% variable 1, and beta be variable 2. To enforce alpha=0.5 and beta=0.75 at
% node 19, supply the following in diri_list: diri_list = [ 19 0.5 ...  
%                                                          119 0.75] 
%							 
% If there are no diri nodes, supply an empty matrix ([]) as diri_list.

% Copyright 2000 by Michael B. Taylor, all rights reserved.
% Thanks to Zac Chambers for bug fixes.

% check arguments for reasonableness
tdof = size(mat,1);			% total number of degrees of freedom
if size(mat,2)==tdof & prod(size(mat))==tdof^2
  % good, mat is a square tdof by tdof matrix
else
  error('mat must be a square matrix')
end

if size(rhs,1)==tdof & prod(size(rhs))==tdof
  % good, rhs has the proper dimensions
else
  error('rhs must be a column matrix with the same number of rows as mat')
end

if isempty(diri_list)
  % there are no diri nodes
elseif size(diri_list,1)<tdof & size(diri_list,2)==2
  % valid dimensions for diri_list
else
  error('check dimensions of diri_list')
end

if ndof>=1 & mod(tdof,ndof)==0
  nnodes = tdof / ndof;			% number of nodes
else
  error('ndof must divide evenly into the total number of rows of mat')
end

% calculate permutation vector
p_vec = [1:tdof];
p_vec = reshape(p_vec,ndof,nnodes);
p_vec = p_vec';

% handle the case of no diri nodes separately
if isempty(diri_list)
  % skip diri partitioning, solve the reordered system
  Q = mat(p_vec,p_vec) \ rhs(p_vec);
  % restore the original ordering and report the result
  p_vec_undo(p_vec) = [1:tdof];
  Q = Q(p_vec_undo);
  % segregate the nodal variables into columns
  Q = reshape(Q,nnodes,ndof);
else
  % reorder and partion out diri nodes
  diri_nodes = diri_list(:,1);		% a list of nodes which are diri
  diri_vals = diri_list(:,2);		% diri values
  % form a list of reordered, not-diri nodes
  diri_flag = ismember(p_vec, diri_nodes);
  ndro = p_vec(find(diri_flag==0)); 	% a list of not-diri nodes, reordered
  rhs_reduced = rhs(ndro) - mat(ndro, diri_nodes) * diri_vals;
  % solve the reduced, reordered system
  Q_reduced = mat(ndro,ndro) \ rhs_reduced;
  % insert the diri values into the solution
  Q = [Q_reduced; diri_vals];
  % calculate p_vec_undo to restore Q to the original node ordering
  p_vec_undo([ndro;diri_nodes]) = [1:tdof];
  % restore the original ordering
  Q = Q(p_vec_undo);
  % segregate the nodal variables into columns
  Q = reshape(Q,nnodes,ndof);
end


