function res=asres1D(gscalar,escalar,hyp,mexp,lmat,nvar);
%  usage:  asres1d(gscalar,escalar,hyp,mexp,lmat,nvar)
%
%  Assembles 1d finite element residuals.  Assumes that each basis used 
%  in the element library matrix is of the same degree.  Returns the
%  assembled global jacobian. 
%
%  gscalar - global scalar data (default unity)
%  escalar - element data (default unity)
%  hyp     - hypermatrix contraction data  (default unity)
%  mexp    - mesh measure exponent, a scalar
%  lmat    - element library matrix
%  nvar    - nodal variable,  multiplies the square matrix resulting from
%              the application of the other data. 
%
%  Any of the first three arguments may be omitted by supplying an 
%  empty matrix ([]) in their place.    Ones will be inserted for any
%  legally omitted arguments.  nvar may not be omitted.
%
%  mesh data is passed via X1, a global row array containing the location of
%  all grid points, including internal grid points

global X1               % a row array, nodal coordinate data
nnodes = prod(size(X1));% number of nodes in the mesh
enodes = size(lmat,2);  % num of columns in lmat is num of nodes per element
nelements = (nnodes-1)/(enodes-1); % number of elements

%  check data for validity.  Generate error message and exit if problems
%  are found

%  check to see that X1 is strictly increasing.
for i=1:nnodes-1
  if X1(i+1)-X1(i) <= 0,
    % X1 is not increasing, generate error
    error('nodal coordinates must increase from left to right')
  end
end

%  check to see if library matrix has reasonable dimensions
if enodes >=2 & enodes <=5
  % reasonable values, do nothing
else
  error('library matrix must have between 2  and 5  columns')
end

% check to see that mesh has proper number of nodes for basis chosen
if mod(nnodes-1,enodes-1)==0
  % valid mesh, do nothing
else
  error('number of nodes inconsistent with degree of basis chosen')
end

% check to see that library matrix has proper number of rows
if size(lmat,2)==size(lmat,1)
  % valid size for square matrix
elseif size(lmat,2)^2==size(lmat,1)
  % proper size for hyper-matrix
else
  error('library (hyper)matrix dimensions inconsistent')
end

% check to see that gscalar is a scalar
if isempty(gscalar)
  % empty is a valid value for gscalar
else
  if size(gscalar,1)==1 & size (gscalar,2)==1
    % gscalar is indeed scalar
  else
    error('the first argument must be a scalar')
  end
end

% check the dimensions of escalar
if isempty(escalar)
  % empty is a valid value for escalar
elseif prod(size(escalar))==nelements
    % good, escalar has one entry for each element
else
    error('the second argument must have exactly one entry per element')
end

% check the dimensions of hyp
if isempty(hyp)
  % empty is a valid value for hyp, generate column of ones
  hyp = ones(nnodes,1);
end

% check the number of entries in hyp
if prod(size(hyp))==nnodes
  % good, hyp has one entry for each node
  hyp = hyp(:);				% put hyp in column form
else
  error('the third argument must have exactly one entry per node')
end

% check to make sure that there is a value for the mesh measure exponent
if isempty(mexp)
  error('mesh measure exponent must not be empty')
end

% check to make sure that mexp is a scalar
if prod(size(mexp))==1,
  % good, mexp is a scalar
else
  error('mesh measure exponent must be input as a scalar')
end

% check to be sure that mexp has a reasonable value
if mexp < -4 | mexp > 4
  error('mesh measure exponent must be between -4 and 4')
end

% check the number of entries in nvar
if prod(size(nvar))==nnodes
  % good, nvar has one entry for each node
  nvar = nvar(:);			% put nvar into column form
elseif isempty(nvar)
  % empty matrix is a valid entry for nvar
else
  error('the last argument must have exactly one entry per node')
end

%  end of argument checks

%  apply the global scalar to the library matrix, provide for ommited arg
if isempty(gscalar),
  % do nothing
else
  lmat = gscalar * lmat;
end

%  provide for empty escalar argument
if isempty(escalar),
   escalar = ones(nelements,1);
end

%  element loops for scalar calculations
%  combine mesh measure data with element scalar data
mleft = 1;   %  initialize node addresses for mesh measure calculation
mright = enodes;
if mexp == 1,
  for i=1:nelements
    escalar(i) = escalar(i) * (X1(mright) - X1(mleft));
    mright = mright + enodes - 1;
    mleft = mleft + enodes - 1;
  end
elseif mexp == -1,
  for i=1:nelements
    escalar(i) = escalar(i) / (X1(mright) - X1(mleft));
    mright = mright + enodes - 1;
    mleft = mleft + enodes - 1;
  end
elseif mexp ~= 0,
  for i=1:nelements
    escalar(i,1) = escalar(i,1) * (X1(mright) - X1(mleft))^mexp;
    mright = mright + enodes - 1;
    mleft = mleft + enodes - 1;
  end
end
%  do nothing for the case of mexp == 0  

%  reserve memory for the assembled residual.
res = zeros(nnodes,1); 

%  element loops for matrix calculations
%  one loop for library hyper-matrices, one for square lib matrices

%  initialize left and right node addresses for current element
mleft = 1;
mright = enodes;
if size(lmat,2)==size(lmat,1); 
  for k=1:nelements, % loop for square library matrices
    elres = lmat * escalar(k) * nvar(mleft:mright);% element residual
    res(mleft:mright) = res(mleft:mright) + elres;
    % update left and right node addresses for the next element
    mleft = mleft + enodes - 1;
    mright = mright + enodes - 1;
  end 
else,
  for k=1:nelements, % loop for library hyper-matrices
    ehyp = hyp(mleft:mright);   %  get part of hyp for current element
    for i=1:enodes; % contract hyper-matrix
      for j=1:enodes;
           emat(i,j) = ehyp(:)' * lmat(1+(i-1)*enodes:i*enodes,j) ;
	 % emat(i,j) = lmat(1+(i-1)*enodes:i*enodes,j) * ehyp;
      end
    end
    elres = emat * escalar(k) * nvar(mleft:mright); % element residual
    res(mleft:mright) = res(mleft:mright) + elres;  % assembly
    % update left and right node addresses for the next element
    mleft = mleft + enodes - 1;
    mright = mright + enodes - 1;
  end
end
