function res=asres(gscalar,escalar,mexp,emat,nvar,elcon,DETe,nnodes)
% usage:  res=asres(gscalar,escalar,mexp,emat,nvar,elcon,DETe,nnodes)
%
% Assembles residuals for multi-dimensional elements
% 
% gscalar - a single scalar constant that multiplies the whole jacobian.  
%           Data specification may be omitted by via an empty matrix ([]).
% escalar - column array of element scalars. Each entry multiplies a specific
%           element matrix.  Data specification may be omitted by via an 
%           empty matrix ([]).
% mexp -    mesh measure exponent, a scalar.  Each element matrix is 
%           multiplied by DETe^mexp before assembly, where DETe is 
%           the element determinant.
% emat -    element matrix.  May be a npe by npe master element matrix or a
%           npe by npe by nelements array where emat(:,:,i) is an element
%           matrix for the ith element. (npe is nodes per element, 3 for 
%           2D linear triangles, 8 for 3D tri-linear bricks, ect)
% nvar -    nodal variable. diag(nvar) post-multiplies the whole jacobian.
%           May be omitted via an empty matrix.
% elcon -   element connectivity table. Contains one row for each
%           element.  The ith row is an ordered list of node numbers
%           comprising the ith element.
% DETe -    A column array of element determinants, with the ith entry
%           corresponding to the ith element.
% nnodes -  total number of nodes
%
% Copyright 2000 by Michael B. Taylor

% transpose elcon
elcon = elcon';

% find the total number of nodes.
nelements = size(elcon,2);

% find the number of nodes per element
npe = size(elcon,1);

% Dimension indx, jndx, and sval.  A given row of [indx,jndx,sval]
% describes the i index, j index, and numerical value of a non-zero entry 
% in the global jacobian matrix.  indx, jndx, and sval will later be used 
% as arguments to sparse, resulting in res.  Since res will be a column
% matrix, jndx is all ones.
indx = elcon;
jndx = ones(npe,nelements);
sval = zeros(npe,nelements);

% handle gscalar and escalar data
if isempty(escalar),
  escalar = ones(nelements,1);
elseif prod(size(escalar)) == nelements
  % escalar is dimensioned properly
else
  error('Check dimensions of escalar and elcon.')
end

if isempty(gscalar),
  % do nothing
elseif prod(size(gscalar)) == 1
  escalar = escalar * gscalar;
else
  error('gscalar should be a scalar or an empty matrix')
end

% handle mesh measure data
if mexp == -1,
  escalar = escalar ./ DETe;
elseif mexp == 0,
  % do nothing
elseif mexp == 1
  escalar = escalar .* DETe;
else
  escalar = escalar .* (DETe .^ mexp);
end

% check for valid nvar
if prod(size(nvar))==nnodes
  % nvar data is present and properly dimensioned
  % compute element nvar data, nvar_e (one column per element)
  nvar_e = nvar(elcon);
else
  error('nvar must have one entry per node')
end

% handle emat data.  select algorithm based on hyper-matrix vs square
% matrix, one master element matrix for all elements vs one element
% matrix per element
if size(emat,1)==npe & size(emat,2) == npe & size(emat,3)==1
  % square matrix, a single matrix for all elements
  nvar_e = nvar_e .* (ones(npe,1) * escalar(:)');
  sval = emat * nvar_e;
elseif size(emat,1)==npe & size(emat,3)==nelements
  % square matrix, one matrix for each element
  %
  % A straightforward contraction of each element matrix would
  % require an element loop and be slow (in Matlab).  Instead, expand
  % contraction data by an outer product, and apply it to the
  % element matrices via an array product.  Sum the rows of the
  % element matrices 
  nvar_e = nvar_e .* (ones(npe,1) * escalar(:)'); % nodal and
                                                  % element data
  emat = reshape(emat,npe,npe*nelements);
  emat = emat .* (ones(npe,1) * nvar_e(:)'); % array product
  emat = reshape(emat,npe,npe,nelements);
  sval = sum(emat,2);			     % sum the rows
  sval = sval(:);
else
  error('check dimensions of emat and elcon')
end

% assemble res
res = full(sparse(indx,jndx,sval,nnodes,1));
