function jac=asjac(gscalar,escalar,mexp,emat,nvar,elcon,DETe,nnodes)
% usage:  jac=asjac(gscalar,escalar,hyp,mexp,emat,nvar,elcon,DETe,nnodes)
%
% Assembles jacobians for multi-dimensional elements
% 
% gscalar - a single scalar constant that multiplies the whole jacobian.  
%           Data specification may be omitted by via an empty matrix ([]).
% escalar - column array of element scalars. Each entry multiplies a specific
%           element matrix.  Data specification may be omitted by via an 
%           empty matrix ([]).
% mexp -    mesh measure exponent, a scalar.  Each element matrix is 
%           multiplied by DETe^mexp before assembly, where DETe is 
%           the element determinant.
% emat -    element matrix.  May be a npe by npe master element matrix or a
%           npe by npe by nelements array where emat(:,:,i) is an element
%           matrix for the ith element. (npe is nodes per element, 3 for 
%           2D linear triangles, 8 for 3D tri-linear bricks, ect)
% nvar -    nodal variable. diag(nvar) post-multiplies the whole jacobian.
%           May be omitted via an empty matrix.
% elcon -   element connectivity table. Contains one row for each
%           element.  The ith row is an ordered list of node numbers
%           comprising the ith element.
% DETe -    An array of element determinants, with the ith entry
%           corresponding to the ith element.
% nnodes -  total number of nodes
%
% The assembled jacobian, jac, is returned in sparse matrix storage.

% This routine is based heavily on the Matlab 'sparse' command.
% According to the Matlab 5.3 documentation on 'sparse', "Any elements of 
% s that have duplicate values of i and j are added together."  This
% behavior is exploited here to do the actual assembly.

% Copyright 2000 by Michael B. Taylor

% find the number of nodes per element
npe = size(elcon,2);

% find the total number of elements.
nelements = size(elcon,1);

indx = elcon(:) * ones(1,npe);
indx = reshape(indx,nelements,npe*npe);
indx = indx';

% indx and jndx will be used with "sparse" at the end of the routine to
% do the assembly.  looking at the output of something like:
% [indx,jndx,sval] = find(ones(4,4))  may shed some light on what is
% going on here.

% handle gscalar and escalar data
if isempty(escalar),
  escalar = ones(nelements,1);
elseif prod(size(escalar)) == nelements
  % escalar is dimensioned properly
else
  error('Check dimensions of escalar and elcon.')
end

if isempty(gscalar),
  % do nothing
elseif prod(size(gscalar)) == 1
  escalar = escalar * gscalar;
else
  error('gscalar should be a scalar or an empty matrix')
end

% handle mesh measure data
if mexp == -1,
  escalar = escalar ./ DETe;
elseif mexp == 0,
  % do nothing
elseif mexp == 1
  escalar = escalar .* DETe;
else
  escalar = escalar .* (DETe .^ mexp);
end

% handle emat data.  select algorithm based on hyper-matrix vs square
% matrix, one master element matrix for all elements vs one element
% matrix per element
if size(emat,1)==npe & size(emat,2)==npe & size(emat,3)==1
  % square matrix, a single emat matrix for all elements
  sval = emat(:) * escalar(:)';
elseif size(emat,1)==npe & size(emat,3)==nelements
  % square matrix, one emat matrix for each element
  emat = reshape(emat,npe*npe,nelements);
  % each column of emat now contains element matrix data for one element
  sval = emat .* (ones(npe*npe,1) * escalar(:)');
else
  error('check dimensions of emat and elcon')
end

% transpose elcon
elcon = elcon';

jndx = ones(npe,1) * elcon(:)';

% make jac from indx, jndx, and sval.  run through sparse twice to get
% the final jac to occupy the smallest possible amount of memory
jac = sparse(indx,jndx,sval,nnodes,nnodes);
[indx,jndx,sval] = find(jac);
jac = sparse(indx,jndx,sval,nnodes,nnodes);
%
% handle nvar
if isempty(nvar)
  % do nothing
elseif prod(size(nvar))==nnodes
  % nvar data is present and properly dimensioned, process it
  jac = jac * spdiags(nvar(:),0,nnodes,nnodes);
else
  error('nvar must be empty or have one entry per node')
end

