/*
    Messages.h, Copyright 11.2.97 (c) by Christian Conrad,
    Swiss Federal Institute of Technology,
    Computer Engineering and Networks Laboratory.

    TOPSY -- A Teachable Operating System.
             Implementation of a tiny and simple
             micro kernel for teaching purposes.

    For further information, please visit http://www.tik.ee.ethz.ch/~topsy

    This software is provided under the terms of the GNU General Public Licence.

    A full copy of the GNU GPL is provided in the file COPYING found in the 
    development root of Topsy.
*/
/*
 	PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

	
	File:                  $Source: /usr/drwho/vault/cvs/topsy/Topsy/Topsy/Messages.h,v $
 	Author(s):             Christian Conrad
 	Affiliation:           ETH Zuerich, TIK
 	Version:               $Revision: 1.31 $
 	Creation Date:         11.2.97
 	Last Date of Change:   $Date: 2000/04/03 17:45:31 $      by: $Author: gfa $
	
	
	$Log: Messages.h,v $
	Revision 1.31  2000/04/03 17:45:31  gfa
	*** empty log message ***
	
	Revision 1.30  1999/12/13 21:48:37  ruf
	GNU General Public Licence Update
	
	Revision 1.29  1999/01/07 16:15:55  cjeker
	removed lightWeight Threads and added new syscall tmGetIDInfo
	
	Revision 1.28  1998/06/05 14:40:29  gfa
	get/set time
	
	Revision 1.27  1998/03/31 15:57:10  gries
	*** empty log message ***

	Revision 1.26  1998/03/31 15:54:23  gries
	IO_Timer messages added

	Revision 1.25  1998/03/26 20:35:05  gfa
	changed tmInfo message for getFirst/Next

	Revision 1.24  1997/04/07 10:12:15  gfa
	removed procedur
	ThreadMainFunction.

 * Revision 1.23  97/04/06  18:46:44  gfa
 * added vmCleanup syscall
 * 
 * Revision 1.22  1997/03/28  12:16:18  gfa
 * added thread info enum
 * (as a msg id subtype it is allowed to be here...)
 *
 * Revision 1.21  1997/03/28  12:13:04  gfa
 * removed unnecessary definitions for msg send/recv
 *
 * Revision 1.20  1997/03/28  12:08:46  gfa
 * added tmGetInfo syscall, types and constants
 *
 * Revision 1.19  1997/03/24  20:32:19  gfa
 * removed all traces of hmAlloc/Free...
 *
 * Revision 1.18  1997/03/22  12:51:52  conrad
 * Added generic user message
 *
 * Revision 1.17  1997/03/16  22:13:51  gfa
 * removed address space arg from vmMove
 *
 * Revision 1.16  1997/03/14  11:04:07  gfa
 * changed ANY to be defined as 1 to trigger compiler warnings
 * (gcc is type-unsafe for int and int* when the value suplied is 0 !!!)
 *
 * Revision 1.15  1997/03/13  18:32:01  gfa
 * *** empty log message ***
 *
 * Revision 1.14  1997/03/11  08:18:04  gfa
 * *** empty log message ***
 *
 * Revision 1.13  1997/03/09  20:53:15  gfa
 * *** empty log message ***
 *
 * Revision 1.12  1997/03/04  16:56:48  conrad
 * Suppression of TMExit message
 *
 * Revision 1.11  1997/02/24  21:35:45  conrad
 * incomplete version
 *
 * Revision 1.10  1997/02/23  18:33:06  gfa
 * changed read/write size's type to unsigned
 *
 * Revision 1.9  1997/02/21  14:14:00  conrad
 * Modified int parameter into ThreadArg parameter
 * new TMExitMsg with thread id as field
 *
 * Revision 1.8  1997/02/21  09:17:21  conrad
 * Adding of a new constant ANYMSGTYPE and cosmetics
 *
 * Revision 1.7  1997/02/20  10:18:11  conrad
 * Adding of name field in TMStartMsg
 *
 * Revision 1.6  1997/02/17  14:41:12  gfa
 * added syscall reply message
 *
 * Revision 1.5  1997/02/13  15:49:14  conrad
 * First compilation/linking of complete environment (all modules)
 *
 * Revision 1.4  1997/02/13  07:43:55  conrad
 * Updating TmStartMsg into TMStartMsg, ...
 * Removal of message TMSendReply (use instead of return value)
 * Removal of message IOInitReply (empty message)
 *
 * Revision 1.3  1997/02/12  13:04:59  conrad
 * Adding of a new MessageId (UNKNOWNMSGREPLY)
 *
 * Revision 1.2  1997/02/11  16:29:21  conrad
 * Message structures declaration. Valid for the whole Topsy.
 *
 * Revision 1.1  1997/02/11  16:10:31  conrad
 * Initial revision
 *
*/

#ifndef __MESSAGES_H
#define __MESSAGES_H

#include "Topsy.h"
#include "Memory.h"
#include "IO.h"
#include "Configuration.h"


#define SELF         0         /* info about one self */
#define ANY          1         /* Receiving from any thread */
#define INFINITY    -1         /* Infinite timeout when receiving message */

/*
 * Message ids and errors
 */

typedef enum {
  ANYMSGTYPE,
  
  TM_START, TM_EXIT, TM_YIELD, TM_KILL, 
  TM_STARTREPLY, TM_KILLREPLY, TM_INFO, TM_INFOREPLY,      /* thread manager */

  TM_GETTIME, TM_GETTIMEREPLY, TM_SETTIME, TM_SETTIMEREPLY,
  TM_TIMERSET, TM_TIMERLEFT, TM_TIMERCLEAR, TM_TIMERTICKS, 
  TM_TIMERSETREPLY, TM_TIMERLEFTREPLY, TM_TIMERCLEARREPLY, 
  TM_TIMEREXPIREDREPLY, TM_TIMERTICKSREPLY,                /* time & timer */

  VM_ALLOC, VM_FREE, VM_MOVE, VM_PROTECT,                  /* memory manager */
  VM_ALLOCREPLY, VM_FREEREPLY, VM_MOVEREPLY, VM_PROTECTREPLY,
  VM_CLEANUP,
  
  IO_OPEN, IO_CLOSE, IO_READ, IO_WRITE, IO_INIT,	   /* input/output */
  IO_OPENREPLY, IO_CLOSEREPLY, IO_READREPLY, IO_WRITEREPLY, IO_INITREPLY,
  
  IO_SUBSCRIBE, IO_SUBSCRIBEREPLY, 
  IO_NOTIFY, IO_GETADDR, IO_GETADDRREPLY,               /* async device */
  
  IO_TIMER_START, IO_TIMER_STOP, IO_TIMERREPLY,         /* hw timer (option) */

  UNKNOWN_SYSCALL                                       /* unknown syscalls */
} MessageId;

typedef enum {
  TM_MSGSENDOK, TM_MSGSENDFAILED, TM_MSGRECVOK, TM_MSGRECVFAILED,
  TM_STARTOK, TM_STARTFAILED, TM_KILLOK, TM_KILLFAILED, 
  TM_INFOOK, TM_INFOFAILED, TM_TIMEOK, TM_TIMEFAILED,

  TM_TIMERSETOK, TM_TIMERSETFAILED, TM_TIMERLEFTOK, TM_TIMERLEFTFAILED,
  TM_TIMEREXPIREDOK, TM_TIMEREXPIREDFAILED, TM_TIMERCLEAROK, 
  TM_TIMERCLEARFAILED, TM_TIMERTICKSOK, TM_TIMERTICKSFAILED,

  VM_ALLOCOK, VM_ALLOCFAILED, VM_FREEOK, VM_FREEFAILED,
  VM_MOVEOK, VM_MOVEFAILED, VM_PROTECTOK, VM_PROTECTFAILED,
  VM_CLEANUPOK, VM_CLEANUPFAILED,

  IO_OPENOK, IO_OPENFAILED, IO_CLOSEOK, IO_CLOSEFAILED,
  IO_READOK, IO_READFAILED, IO_WRITEOK, IO_WRITEFAILED,
  IO_INITOK, IO_INITFAILED,
  
  IO_SUBSCRIBEOK, IO_SUBSCRIBEFAILED
} MessageError;

typedef enum {SYSCALL_SEND_OP, SYSCALL_RECV_OP} MsgOpCode;

/* subtype for various infos on threads */ 
typedef enum {SPECIFIC_ID, GETFIRST, GETNEXT, THREAD_BY_NAME, GET_ID_INFO} ThreadInfoKind; 

typedef enum {Info_RUNNING, Info_READY, Info_BLOCKED} ThreadInfoStatus;

/* thread structure for users */
typedef struct ThreadInfo_t {        
    ThreadId tid;
    ThreadId ptid ;
    ThreadInfoStatus status ;
    char name[MAXNAMESIZE] ;
} ThreadInfo;


/*
 * Message structures: Threads
 */
typedef struct TMStartMsg_t { 	     /* Message to create a thread */
    ThreadMainFunction fctnAddress;
    ThreadArg parameter;
    char *name;
} TMStartMsg;

typedef struct TMStartReplyMsg_t {   /* Message to create a thread (reply) */
    ThreadId newId;    
    MessageError errorCode;
} TMStartReplyMsg;

typedef struct TMKillMsg_t { 	     /* Message to kill a thread */
    ThreadId id;
} TMKillMsg;

typedef struct TMKillReplyMsg_t {    /* Message to kill a thread (reply) */
    MessageError errorCode;
} TMKillReplyMsg;

typedef struct TMExitMsg_t { 	     /* Message to "exit" a thread */
    ThreadId id;
} TMExitMsg;

typedef struct TMGetInfoMsg_t {      /* Message to get info about threads */
    ThreadId about;
    ThreadInfoKind kind;
    ThreadInfo* infoPtr;
} TMGetInfoMsg;

typedef struct TMGetInfoReplyMsg_t { /* Message to get info (reply) */
    MessageError errorCode;
    long int info[2];
} TMGetInfoReplyMsg;

typedef struct TMTimeMsg_t {         /* Message to get/set real time */
    MessageError errorCode;
    unsigned long int seconds;
    unsigned long int microSeconds;
} TMTimeMsg;


/*
 * Message structures: Memory
 */
typedef struct VMAllocMsg_t {        /* Message to get virtual memory */
    unsigned long int size;
} VMAllocMsg;

typedef struct VMAllocReplyMsg_t {   /* Message to get virtual memory (reply) */
    MessageError errorCode;
    Address address;
} VMAllocReplyMsg;

typedef struct VMFreeMsg_t {         /* Message to free virtual memory */
    Address address;
} VMFreeMsg;

typedef struct VMFreeReplyMsg_t {    /* Message to free virtual memory (reply) */
    MessageError errorCode;
} VMFreeReplyMsg;

typedef struct VMMoveMsg_t {         /* Message to move memory region */
    Address address;
    ThreadId newOwner;
} VMMoveMsg;

typedef struct VMMoveReplyMsg_t {    /* Message to move memory region (reply) */
    MessageError errorCode;
    Address address;
} VMMoveReplyMsg;

typedef struct VMProtectMsg_t {      /* Message to protect memory region */
    Address startAddress;
    unsigned long int size;
    ProtectionMode pmode;
} VMProtectMsg;

typedef struct VMProtectReplyMsg_t { /* protect memory region (reply) */
    MessageError errorCode;
} VMProtectReplyMsg;

typedef struct VMCleanupMsg_t {      /* cleanup memory regions for thread */
    ThreadId threadId;
} VMCleanupMsg;


/*
 * Message structures: IO
 */
typedef struct IOOpenMsg_t {
    long int deviceNumber;
} IOOpenMsg;

typedef struct IOOpenReplyMsg_t {
    MessageError errorCode;
    long int deviceThreadId;
} IOOpenReplyMsg;

typedef struct IOCloseMsg_t {
    long int deviceThreadId;
} IOCloseMsg;

typedef struct IOCloseReplyMsg_t {
    MessageError errorCode;
} IOCloseReplyMsg;

typedef struct IOReadMsg_t {
    Address buffer;
    unsigned long int size;
} IOReadMsg;

typedef struct IOReadReplyMsg_t {
    MessageError errorCode;
    unsigned long int bytesRead;
} IOReadReplyMsg;

typedef struct IOWriteMsg_t {
    Address buffer;
    unsigned long int size;
} IOWriteMsg;

typedef struct IOWriteReplyMsg_t {
    MessageError errorCode;
    unsigned long int bytesWritten;
} IOWriteReplyMsg;

typedef struct IOInitReplyMsg_t {
    MessageError errorCode;
} IOInitReplyMsg;

typedef struct { /* Message for hw-timer */
    int period;
} IOTimerMsg;

typedef struct IOSubscribeMsg_t {  /* async device */
} IOSubscribeMsg;

typedef struct IOSubscribeReplyMsg_t {  /* async device */
    MessageError errorCode;
} IOSubscribeReplyMsg;

typedef struct IONotifyMsg_t {  /* async device */
    long int deviceNumber;
    unsigned long int length;
} IONotifyMsg;

typedef struct IOGetAddrReplyMsg_t {  /* async device */
    unsigned char addr[12];           /* Ethernet: first 6 bytes */
} IOGetAddrReplyMsg;


/*
 * Message structures: Generic messages
 */
typedef struct SyscallReply_t {
    MessageError errorCode;
} SyscallReply;

typedef struct UserMessage_t {  
    void* p1;
    void* p2;
    void* p3;
} UserMessage;



typedef union SpecMsg_u {            /* Union type of all possible messages */
    UserMessage userMsg;

    TMStartMsg tmStart; TMStartReplyMsg tmStartReply;
    TMKillMsg tmKill; TMKillReplyMsg tmKillReply;
    TMGetInfoMsg tmInfo; TMGetInfoReplyMsg tmInfoReply;
    TMTimeMsg tmTime;
    
    VMAllocMsg vmAlloc; VMAllocReplyMsg vmAllocReply;
    VMFreeMsg vmFree; VMFreeReplyMsg vmFreeReply;
    VMMoveMsg vmMove; VMMoveReplyMsg vmMoveReply;
    VMProtectMsg vmProtect; VMProtectReplyMsg vmProtectReply;
    VMCleanupMsg vmCleanup;

    IOOpenMsg ioOpen; IOOpenReplyMsg ioOpenReply;
    IOCloseMsg ioClose; IOCloseReplyMsg ioCloseReply;
    IOReadMsg ioRead; IOReadReplyMsg ioReadReply;
    IOWriteMsg ioWrite; IOWriteReplyMsg ioWriteReply;
                        IOInitReplyMsg ioInitReply;
    IOTimerMsg ioTimer;

    IOSubscribeMsg ioSubscribe; IOSubscribeReplyMsg ioSubscribeReply;
    IONotifyMsg ioNotify; IOGetAddrReplyMsg ioGetAddrReply;
    
    SyscallReply syscallReply;
} SpecMessage;

typedef struct Message_t {     /* Main message structure */
    ThreadId from;		   /* Sender of the message */
    MessageId id;		   /* Type of the message */
    SpecMessage msg;		   /* Specific message contents */
} Message;


#endif __MESSAGES_H
