/*
    TMError.c, Copyright  (c) by Christian Conrad,
    Swiss Federal Institute of Technology,
    Computer Engineering and Networks Laboratory.

    TOPSY -- A Teachable Operating System.
             Implementation of a tiny and simple
             micro kernel for teaching purposes.

    For further information, please visit http://www.tik.ee.ethz.ch/~topsy

    This software is provided under the terms of the GNU General Public Licence.

    A full copy of the GNU GPL is provided in the file COPYING found in the 
    development root of Topsy.
*/
/*
	
	File:                  $Source: /usr/drwho/vault/cvs/topsy/Topsy/Threads/mips/TMError.c,v $
 	Author(s):             Christian Conrad
 	Affiliation:           ETH Zuerich, TIK
 	Version:               $Revision: 1.4 $
 	Creation Date:         
 	Last Date of Change:   $Date: 1999/12/13 21:48:35 $      by: $Author: ruf $
	
	
	$Log: TMError.c,v $
	Revision 1.4  1999/12/13 21:48:35  ruf
	GNU General Public Licence Update
	
	Revision 1.3  1998/04/07 15:31:38  gfa
	added printreg
	
	Revision 1.2  1997/04/23 09:10:28  gfa
	formatting, comments adjusted

 * Revision 1.1  1997/04/08  20:29:39  conrad
 * Initial revision
 *
*/

#include "TMError.h"
#include "Threads.h"
#include "Messages.h"
#include "TMHal.h"
#include "TMScheduler.h"


static void tmReservedInstructionError(ThreadId currentThread);
static void tmCPunusableError(ThreadId currentThread);
static void tmOverflowError(ThreadId currentThread);
static void tmBreakpointError(ThreadId currentThread);


void tmInstallErrorHandlers(void)
{
    /* Vital for correct operations
     */
    tmSetExceptionHandler(SYSCALL, syscallExceptionHandler);
    tmSetExceptionHandler(HARDWARE_INT, hwExceptionHandler);
    tmSetInterruptHandler(CLOCKINT_0, tmClockHandler, NULL);

    /* Error handling for faulting threads
     */
    tmSetExceptionHandler(RES_INSTR, tmReservedInstructionError);
    tmSetExceptionHandler(CP_UNUSABLE, tmCPunusableError);
    tmSetExceptionHandler(OVERFLOW, tmOverflowError);
    tmSetExceptionHandler(BREAKPOINT, tmBreakpointError);
}


static void tmError(ThreadId currentThread, char* errorString)
{
    Message msg;
    
    msg.id = TM_KILL;
    msg.from = TMTHREADID;
    msg.msg.tmKill.id = currentThread;
    
    ERROR(errorString);
    printRegisters();

    /* immediate in-kernel delivery without syscall
     * kSend does a schedule after the message arrived
     */
    kSend(TMTHREADID, &msg); 
}

static void tmReservedInstructionError(ThreadId currentThread)
{
    tmError(currentThread, "reserved instruction");
}

static void tmCPunusableError(ThreadId currentThread)
{
    tmError(currentThread, "CP unusable");
}

static void tmOverflowError(ThreadId currentThread)
{
    tmError(currentThread, "arithmetic overflow");
}

static void tmBreakpointError(ThreadId currentThread)
{
    tmError(currentThread, "breakpoint");
}

