/*
    TMHal.h, Copyright  (c) by Lukas Ruf, lr@lpr.ch,
    Swiss Federal Institute of Technology,
    Computer Engineering and Networks Laboratory.

    TOPSY -- A Teachable Operating System.
             Implementation of a tiny and simple
             micro kernel for teaching purposes.

    For further information, please visit http://www.tik.ee.ethz.ch/~topsy

    This software is provided under the terms of the GNU General Public Licence.

    A full copy of the GNU GPL is provided in the file COPYING found in the 
    development root of Topsy.
*/
/*
	
	File:                  $Source: /usr/drwho/vault/cvs/topsy/Topsy/Threads/ia32/TMHal.h,v $
 	Author(s):             Lukas Ruf, lr@lpr.ch
 	Affiliation:           ETH Zuerich, TIK
 	Version:               $Revision: 1.4 $
 	Creation Date:         
 	Last Date of Change:   $Date: 1999/12/13 21:48:34 $      by: $Author: ruf $
	
	
	$Log: TMHal.h,v $
	Revision 1.4  1999/12/13 21:48:34  ruf
	GNU General Public Licence Update
	
	Revision 1.3  1999/10/23 19:07:15  jeker
	added Makefile.SimOS.mips and did some code cleanup
	
	Revision 1.2  1999/06/06 20:55:18  jeker
	putting everything together for Topsy 2.0

	Revision 1.1  1999/05/13 17:05:50  jeker
	Initial revision

*/
#ifndef __TMHAL_H
#define __TMHAL_H

#include "tm-include.h"
#include "cpu.h"
#include "Exception.h"

#define MAXNBOFEXCEPTIONS    32
#define MAXNBOFINTERRUPTS     8

/* THREADRETURNADDR is defined inside TMHal.h to allow portability.... :-)
   Stack-Layout on Function entry from C on i386:
    void foo(ARG1,ARG2,...,ARGn);
    Size  Function
      4     ARGn
      ..    ...
      4     ARG2
      4     ARG1
      4     EIP after CALL statement
   ESP points to the lowest byte of the above noted EIP
   
   As in Topsy v1.x (where this port is based on) only two thread arguments
   are provided, the "AutomaticThreadExit" is located after ARG2, i.e. ESP+12
   
   Note, this ThreadReturnAddr is relative to the Stack Pointer
 */
#define THREADRETURNADDR    12


/* Processor Status Register Flags: Called Extended Processor Flags, eflags */

#define PSL_C     0x00000001  /* carry bit */
#define PSL_PF    0x00000004  /* parity bit */
#define PSL_AF    0x00000010  /* bcd carry bit */
#define PSL_Z     0x00000040  /* zero bit */
#define PSL_N     0x00000080  /* negative bit */
#define PSL_T     0x00000100  /* trace enable bit */
#define PSL_I     0x00000200  /* interrupt enable bit */
#define PSL_D     0x00000400  /* string instruction direction bit */
#define PSL_V     0x00000800  /* overflow bit */
#define PSL_IOPL  0x00003000  /* i/o privilege level */
#define PSL_NT    0x00004000  /* nested task bit */
#define PSL_RF    0x00010000  /* resume flag bit */
#define PSL_VM    0x00020000  /* virtual 8086 mode bit */
#define PSL_AC    0x00040000  /* alignment checking */
#define PSL_VIF   0x00080000  /* virtual interrupt enable */
#define PSL_VIP   0x00100000  /* virtual interrupt pending */
#define PSL_ID    0x00200000  /* identification bit */


#define PSL_RESERVED          0x00000002

#define PSL_KERNEL            (PSL_RESERVED | PSL_I)
#define PSL_USER              (PSL_RESERVED | PSL_I)

#define STATUS_INT_ENABLE_KERNEL_PREV PSL_KERNEL
#define STATUS_INT_ENABLE_USER_PREV   PSL_USER


/*
 * System stack frames.
 */

/*
 * Exception/Trap Stack Frame
 */

typedef struct ProcContext_t {
  int tf_gs;      /* I include even the new gs and fs selectors, even neither */
  int tf_fs;      /* FreeBSD 2.2.1 nor Linux 2.0.32 do. You never know :-)    */
  int tf_es;      
  int tf_ds;
  int tf_trapno;
  int tf_edi;
  int tf_esi;
  int tf_ebp;
  int tf_temp_esp;  /* this is esp before the pushal was executed */
  int tf_ebx;
  int tf_edx;
  int tf_ecx;
  int tf_eax;
  /* NOTE: To this location tf_temp_esp is pointing to :-) */
  /* below portion defined in 386 hardware */
  int tf_err;   /* tf_err is defined here so all kind ISRs can use the same */
  int tf_eip;   /*        structure to save the current processor context   */
  int tf_cs;
  int tf_eflags;
  /* below only when crossing rings (e.g. user to kernel) */
  int tf_esp; /* user stack pointer */
  int tf_ss;  /* user stack segment */
} ProcContext;


/* Saving and restoring current thread context (assembler procedures) */
void saveContext(ProcContextPtr contextPtr);
void restoreContext(ProcContextPtr contextPtr);

/* operations on the thread context are machine dependent and 
 * encapsulated in the following functions. On CISC machines
 * most of these ops must be implemented using the current SP of
 * the context.
 */

/*
intDispatcher: LR 980501 / is called from IRQ.c --> needs definition
 */
void intDispatcher(ProcContextPtr frame);

/* LR 19980609: set Machine Dependencies on Proc Context */
void tmSetMachineDependentRegisters(ProcContextPtr contextPtr, AddressSpace space);

void tmSetReturnValue(ProcContextPtr contextPtr, Register value);
void tmSetStackPointer(ProcContextPtr contextPtr, Register value);
void tmSetReturnAddress(ProcContextPtr contextPtr, Register value);
void tmSetProgramCounter(ProcContextPtr contextPtr, Register value);
void tmSetStatusRegister(ProcContextPtr contextPtr, Register value);
void tmSetFramePointer(ProcContextPtr contextPtr, Register value);
void tmSetArgument0(ProcContextPtr contextPtr, Register value);
void tmSetArgument1(ProcContextPtr contextPtr, Register value);

void enableInterruptInContext( InterruptId intId, ProcContextPtr contextPtr);
void disableInterruptInContext( InterruptId intId, ProcContextPtr contextPtr);
void enableAllInterruptsInContext( ProcContextPtr contextPtr);

/* installs code by copying generalExceptionHandler()'s and 
 * UTLBMissHandler()'s code at certain mips specific addresses
 */
void tmInstallExceptionCode();

/* Syscall exception handler (set per default as exception handler) */
void syscallExceptionHandler(ThreadId threadId);

/* Automatic exit of a thread when stack underflow occurs */
void automaticThreadExit();
void endAutomaticThreadExit();

void msgAdjust(ThreadPtr pFromOrToThread, Message *msg, Boolean ToKernel);

void powerSaver();

#endif __TMHAL_H

